# -*- coding: utf-8 -*-
"""
Daniel Calderon S.

file: botellas_control.py
-------------------------
Dibuja botellas distribuidas aleatoriamente sobre el plano.
Permite mover luces y cámara.
Utiliza clase Vector3D para modelar la botella y los movimientos.

Control del programa:
1: visualiza o no los ejes de coordenadas
2: visualiza o no como malla de alambre
3: alterna entre smooth y flat
q,a: rotar botella
w,s: mover cámara
e,d: mover luz1
z: apaga/enciende luz0
ESC: terminar
"""

import pygame
from pygame.locals import *

from OpenGL.GL import *
from OpenGL.GLU import *
from OpenGL.GLUT import *

from AuxiliaryFunctions import *
import Vector3D as v3d

from math import pi
from random import uniform

#####################################################################

def init_pygame((w,h), title=""):
    pygame.init()
    pygame.display.set_mode((w,h), OPENGL|DOUBLEBUF)
    pygame.display.set_caption(title)

def init_opengl((w,h)):
    reshape((w,h))
    init()
 
def init():
    # setea el color de fondo
    glClearColor(0.0, 0.0, 0.0, 1.0)
	
    # se habilitan las transparencias
    glEnable(GL_BLEND)
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
    # el color debe incluir la cuarta componente, alpha
    # alpha=1  --> objeto totalmente opaco
    # alpha=0  --> opbjeto totalmente transparente
 
    glEnable(GL_DEPTH_TEST)
    glDepthFunc(GL_LEQUAL)
    #glDepthFunc(GL_LESS)
    
    # normaliza las normales luego del escalamiento.
    glEnable(GL_NORMALIZE)
	
def reshape((width, height)):
    if height == 0:
        height = 1
    glViewport(0, 0, width, height)
    glMatrixMode(GL_PROJECTION)
    glLoadIdentity()
    gluPerspective(60.0, float(width)/float(height), 0.1, 20000.0)
    #glOrtho(-w,w,-h,h,1,20000)
    glMatrixMode(GL_MODELVIEW)
    glLoadIdentity()

#####################################################################

def initLights():    
    # se configura la fuente de luz 0 y se habilita    
    light = GL_LIGHT0
    position = v3d.Vector( 1500.0, 0.0, 0.0)
    ambient = [ 1.0, 0.2, 0.2, 1.0]
    specular = [ 1.0, 1.0, 1.0, 1.0]
    diffuse = [ 1.0, 1.0, 1.0, 1.0]

    constant_attenuation = 1.0
    linear_attenuation = 0.0
    quadratic_attenuation = 0.0

    spot_cutoff = 180.0
    spot_direction = v3d.Vector(-1.0, 0.0, 0.0)
    spot_exponent = 0.0
    
    enable = True
    
    luz0 = Luz(light,position,ambient,specular,diffuse,constant_attenuation,linear_attenuation,quadratic_attenuation,spot_cutoff,spot_direction,spot_exponent,enable)
    
    # se configura la fuente de luz 1 y se habilita
    light = GL_LIGHT1
    position = v3d.Vector( 0.0, 0.0, 1500.0)
    ambient = [ 1.0, 0.2, 0.2, 1.0]
    specular = [ 1.0, 1.0, 1.0, 1.0]
    diffuse = [ 1.0, 1.0, 1.0, 1.0]

    constant_attenuation = 0.2
    linear_attenuation = 0.0
    quadratic_attenuation = 0.0

    spot_cutoff = 70.0
    spot_direction = v3d.Vector( 0.0, 0.0, -1.0)
    spot_exponent = 0.5
    
    enable = True
    
    luz1 = Luz(light,position,ambient,specular,diffuse,constant_attenuation,linear_attenuation,quadratic_attenuation,spot_cutoff,spot_direction,spot_exponent,enable)
    
    # Habilito iluminación
    glEnable(GL_LIGHTING)
    
    return luz0,luz1

#####################################################################

def listaBotella(n):

    #iniciar lista
    lista = glGenLists(1)
    glNewList(lista,GL_COMPILE)
    
    dang = pi/float(n)
    
    # triang(d,a0,a1) - triang(d,a1,a2)
    # cuad(a1,a0,b0,b1) - cuad(a2,a1,b1,b2)
    # cuad(b1,b0,c0,c1) - cuad(b2,b1,c1,c2)
    # cuad(c1,c0,d0,d1) - cuad(c2,c1,d1,d2)
    # traing(d1,d0,e) - triang(d2,d1,e)
    
    d = v3d.Vector(0.0,0.0,1.0)
    e = v3d.Vector(0.0,0.0,0.0)
    
    nd = v3d.Vector(0.0,0.0,1.0)
    ne = v3d.Vector(0.0,0.0,-1.0)
    
    a0 = v3d.VectorCilindricas(0.2,0.0,1.0)
    b0 = v3d.VectorCilindricas(0.2,0.0,0.8)
    c0 = v3d.VectorCilindricas(0.5,0.0,0.6)
    d0 = v3d.VectorCilindricas(0.5,0.0,0.0)
    
    a1 = v3d.VectorCilindricas(0.2,dang,1.0)
    b1 = v3d.VectorCilindricas(0.2,dang,0.8)
    c1 = v3d.VectorCilindricas(0.5,dang,0.6)
    d1 = v3d.VectorCilindricas(0.5,dang,0.0)
    
    a2 = v3d.VectorCilindricas(0.2,2*dang,1.0)
    b2 = v3d.VectorCilindricas(0.2,2*dang,0.8)
    c2 = v3d.VectorCilindricas(0.5,2*dang,0.6)
    d2 = v3d.VectorCilindricas(0.5,2*dang,0.0)
    
    nA0 = v3d.normal(d,a0,a1)
    nB0 = v3d.normal(a1,a0,b0)
    nC0 = v3d.normal(b1,b0,c0)
    nD0 = v3d.normal(c1,c0,d0)
    nE0 = v3d.normal(d1,d0,e)
    
    nA1 = v3d.normal(d,a1,a2)
    nB1 = v3d.normal(a2,a1,b1)
    nC1 = v3d.normal(b2,b1,c1)
    nD1 = v3d.normal(c2,c1,d1)
    nE1 = v3d.normal(d2,d1,e)
    
    na0 = v3d.promediar([nA0,nA1,nB0,nB1])
    nb0 = v3d.promediar([nB0,nB1,nC0,nC1])
    nc0 = v3d.promediar([nC0,nC1,nD0,nD1])
    nd0 = v3d.promediar([nD0,nD1,nE0,nE1])
    
    ang = dang
    while ang<=2*pi+dang:
    
        a0 = v3d.VectorCilindricas(0.2,ang,1.0)
        b0 = v3d.VectorCilindricas(0.2,ang,0.8)
        c0 = v3d.VectorCilindricas(0.5,ang,0.6)
        d0 = v3d.VectorCilindricas(0.5,ang,0.0)
        
        a1 = v3d.VectorCilindricas(0.2,ang+dang,1.0)
        b1 = v3d.VectorCilindricas(0.2,ang+dang,0.8)
        c1 = v3d.VectorCilindricas(0.5,ang+dang,0.6)
        d1 = v3d.VectorCilindricas(0.5,ang+dang,0.0)
        
        a2 = v3d.VectorCilindricas(0.2,ang+2*dang,1.0)
        b2 = v3d.VectorCilindricas(0.2,ang+2*dang,0.8)
        c2 = v3d.VectorCilindricas(0.5,ang+2*dang,0.6)
        d2 = v3d.VectorCilindricas(0.5,ang+2*dang,0.0)
        
        nA0 = v3d.normal(d,a0,a1)
        nB0 = v3d.normal(a1,a0,b0)
        nC0 = v3d.normal(b1,b0,c0)
        nD0 = v3d.normal(c1,c0,d0)
        nE0 = v3d.normal(d1,d0,e)
        
        nA1 = v3d.normal(d,a1,a2)
        nB1 = v3d.normal(a2,a1,b1)
        nC1 = v3d.normal(b2,b1,c1)
        nD1 = v3d.normal(c2,c1,d1)
        nE1 = v3d.normal(d2,d1,e)
        
        na1 = v3d.promediar([nA0,nA1,nB0,nB1])
        nb1 = v3d.promediar([nB0,nB1,nC0,nC1])
        nc1 = v3d.promediar([nC0,nC1,nD0,nD1])
        nd1 = v3d.promediar([nD0,nD1,nE0,nE1])
        
        # 'v' indica que argumentos son del tipo Vector3D
        vdrawTrianglen3(d,a0,a1,nd,na0,na1)
        vdrawQuadn4(a1,a0,b0,b1,na1,na0,nb0,nb1)
        vdrawQuadn4(b1,b0,c0,c1,nb1,nb0,nc0,nc1)
        vdrawQuadn4(c1,c0,d0,d1,nc1,nc0,nd0,nd1)
        vdrawTrianglen3(d1,d0,e,nd1,nd0,ne)
        
        na0 = na1
        nb0 = nb1
        nc0 = nc1
        nd0 = nd1
        
        ang += dang
    
    #terminar lista
    glEndList()
    
    return lista

#####################################################################

class ObjetoSolido:
    def __init__(self,pos,dim,ang,rgb,lista):
        # parámetros iniciales
        self.rgb = rgb # color, considera transparencia
        self.dim = dim # dimensiones: ancho, alto y profundidad
        self.pos = pos # posicion 
        self.ang = ang # rotacion c/r a cada eje.

        # en la lista se encuentra almacenado el dibujo
        self.lista = lista
        
    def dibujar(self):
        #respaldo el sistema de referencia
        glPushMatrix()

        #efectuar transformaciones 
        glTranslatef(self.pos.x,self.pos.y,self.pos.z)
        glRotatef   (self.ang.x,1,0,0)
        glRotatef   (self.ang.y,0,1,0)
        glRotatef   (self.ang.z,0,0,1)
        glScalef    (self.dim.x,self.dim.y,self.dim.z)

        #configurando el material ...
        mat_specular    = [ 0.2, 0.2, 0.2, 1.0 ]
        mat_ambient        = [ 0.0, 0.0, 0.0, 1.0 ]
        mat_diffuse     = self.rgb
        mat_shininess    = [ 10 ] # de 0 a 128
        glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular)
        glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient)
        glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse)
        glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess)
        
        #dibuja el contenido de la lista
        glCallList(self.lista)

        #reestablece el sistema de referencia anterior
        glPopMatrix()
        
#####################################################################

def accion():
    # Limpio buffers
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
    
    #posibilidad de pintar o no los polígonos
    if control.fill_polygons:
        glPolygonMode(GL_FRONT, GL_FILL)
        glPolygonMode(GL_BACK, GL_FILL)
    else:
        glPolygonMode(GL_FRONT, GL_LINE)
        glPolygonMode(GL_BACK, GL_LINE)
    
    # posibilidad de mostrar o no los ejes
    if control.show_axes:
        glDisable(GL_LIGHTING)
        glCallList(axes)
        glEnable(GL_LIGHTING)
    
    # smooth o flat
    if control.smooth_flat:
        glShadeModel(GL_SMOOTH)
    else:
        glShadeModel(GL_FLAT)
        
    # rotar botella central
    if control.q:
        bot.ang = v3d.sumar(bot.ang,v3d.Vector(100.0*dt,0.0,0.0))
        print "bot.ang = ",bot.ang.cartesianas()
    if control.a:
        bot.ang = v3d.sumar(bot.ang,v3d.Vector(-100.0*dt,0.0,0.0))
        print "bot.ang = ",bot.ang.cartesianas()
        
    # rotar cámara en torno a eje z
    if control.w:
        camara.eye = v3d.sumar(camara.at,v3d.rotarFi(v3d.restar(camara.eye,camara.at),dt))
        print "camara.eye = ",camara.eye.cartesianas()
    if control.s:
        camara.eye = v3d.sumar(camara.at,v3d.rotarFi(v3d.restar(camara.eye,camara.at),-dt))
        print "camara.eye = ",camara.eye.cartesianas()
        
    camara.accion()
    
    # luz 0 puede estar encendida o apagada
    luz0.accion()
    
    # mover luz en eje y
    if control.e:
        luz1.position = v3d.sumar(luz1.position,v3d.Vector(0.0,500.0*dt,0.0))
        print "luz1.position = ",luz1.position.cartesianas()
    if control.d:
        luz1.position = v3d.sumar(luz1.position,v3d.Vector(0.0,-500.0*dt,0.0))
        print "luz1.position = ",luz1.position.cartesianas()
    
    luz1.accion()
    
    for b in bs:
        b.dibujar()
    
    bot.dibujar()
    
#####################################################################
# Unidad de control
#####################################################################

class Control:
    def __init__(self):
        self.q = False
        self.w = False
        self.e = False
        self.a = False
        self.s = False
        self.d = False
        
        self.show_axes = True
        self.fill_polygons = True
        self.smooth_flat = True
    
class Camara:
    def __init__(self,eye,at,up):
        self.eye = eye
        self.at  = at
        self.up  = up
    
    def accion(self):
        glLoadIdentity()    
        gluLookAt( self.eye.x, self.eye.y, self.eye.z, self.at.x, self.at.y, self.at.z, self.up.x, self.up.y, self.up.z)


class Luz:
    def __init__(self,light,position,ambient,specular,diffuse,constant_attenuation,linear_attenuation,quadratic_attenuation,spot_cutoff,spot_direction,spot_exponent,enable):
        self.light = light
        self.position = position
        self.ambient = ambient
        self.specular = specular
        self.diffuse = diffuse
        
        self.constant_attenuation = constant_attenuation
        self.linear_attenuation = linear_attenuation
        self.quadratic_attenuation = quadratic_attenuation
        
        self.spot_cutoff = spot_cutoff
        self.spot_direction = spot_direction
        self.spot_exponent = spot_exponent
        
        self.enable  = enable
        
    def accion(self):        
        if self.enable:
            glLightfv(self.light, GL_POSITION, self.position.cartesianas()+[1.0])
            glLightfv(self.light, GL_AMBIENT , self.ambient)
            glLightfv(self.light, GL_SPECULAR, self.specular)
            glLightfv(self.light, GL_DIFFUSE , self.diffuse)
            glLightf(self.light, GL_CONSTANT_ATTENUATION, self.constant_attenuation)
            glLightf(self.light, GL_LINEAR_ATTENUATION, self.linear_attenuation)
            glLightf(self.light, GL_QUADRATIC_ATTENUATION, self.quadratic_attenuation)
            glLightf(self.light, GL_SPOT_CUTOFF, self.spot_cutoff)
            glLightfv(self.light, GL_SPOT_DIRECTION, self.spot_direction.cartesianas())
            glLightf(self.light, GL_SPOT_EXPONENT, self.spot_exponent)
            
            glEnable(self.light)
        else:
            glDisable(self.light)

#####################################################################
# PROGRAMA PRINCIPAL
#####################################################################

if __name__ == "__main__":

    # visibles en todo el archivo
    global dt,eje,bot,bs,luz0,luz1,control,camara

    W=640 #ancho de ventana
    H=480 #alto de ventana

    # inicializando ...
    init_pygame((W,H),"Botella")
    init_opengl((W,H))
    
    # imprime información sobre el Hardware gráfico y
    # la version de OpenGL implementada
    printVersions()

    # configura luces
    luz0,luz1 = initLights()

    # configurando cámara
    eye = v3d.Vector(2000.0,2000.0,2000.0)
    at  = v3d.Vector(0.0, 0.0, 500.0)
    up  = v3d.Vector(0.0,0.0,1.0)
    camara = Camara(eye,at,up)

    # creando objetos ...

    ## Unidad de control
    control = Control()

    ## listas
    axes=axesList(10000)
    listabotella = listaBotella(15)  # dibujo de botella

    ## Botella
    pos = v3d.Vector(0.0,0.0,0.0)        # posición
    dim = v3d.Vector(500.0,500.0,1000.0) # tamaño
    ang = v3d.Vector(0.0,0.0,0.0)        # ángulo respecto de cada eje
    rgb = [0.7,0.7,0.0,0.8]          # color
    bot = ObjetoSolido(pos,dim,ang,rgb,listabotella)

    ## muchas botellas estáticas
    bs = []
    n = 10
    while n>0:
        pos = v3d.VectorCilindricas(uniform(1000.0,2000.0),n*pi/5,0.0)
        sxy = uniform(200.0,500.0)
        dim = v3d.Vector(sxy,sxy,uniform(500.0,1100.0))
        ang = v3d.Vector(0.0,0.0,0.0)
        rgb = [uniform(0.0,1.0),uniform(0.0,1.0),uniform(0.0,1.0),uniform(0.5,1.0)]
        bs = bs + [ObjetoSolido(pos,dim,ang,rgb,listabotella)]
        n -=1

    # tiempo inicial de una iteración
    t0=pygame.time.get_ticks()

    run=True
    while run: # iteracion
        # 0: CONTROL DEL TIEMPO
        t1 = pygame.time.get_ticks()    # tiempo actual
        dt = (t1 - t0)/1000.0           # diferencial de tiempo asociado a la iteración
        t0 = t1                         # actualizar tiempo inicial para siguiente iteración

        # 1: MANEJAMOS EVENTOS DE ENTRADA (TECLADO, MOUSE, ETC.)
        for event in pygame.event.get():

            # cerrar ventana
            if event.type == QUIT:
                run=False

            # presionar tecla
            if event.type == KEYDOWN:
                if event.key == K_ESCAPE:
                    run=False
                if event.key == K_1:
                    control.show_axes = not control.show_axes
                if event.key == K_2:
                    control.fill_polygons = not control.fill_polygons
                if event.key == K_3:
                    control.smooth_flat = not control.smooth_flat
                    if control.smooth_flat:
                        print "smooth"
                    else:
                        print "flat"
                if event.key == K_q:
                    control.q = True
                if event.key == K_w:
                    control.w = True
                if event.key == K_e:
                    control.e = True
                if event.key == K_a:
                    control.a = True
                if event.key == K_s:
                    control.s = True
                if event.key == K_d:
                    control.d = True
                    
                if event.key == K_z:
                    luz0.enable = not luz0.enable
                    
            # soltar tecla
            if event.type == KEYUP:
                if event.key == K_UP:
                    zoom_plus = False
                if event.key == K_DOWN:
                    zoom_minus = False
                    
                if event.key == K_q:
                    control.q = False
                if event.key == K_w:
                    control.w = False
                if event.key == K_e:
                    control.e = False
                if event.key == K_a:
                    control.a = False
                if event.key == K_s:
                    control.s = False
                if event.key == K_d:
                    control.d = False

        # 2: EJECUTAMOS LOGICA DE LA APLICACION
        # y        
        # 3: DIBUJAMOS LOS ELEMENTOS
        accion()
        
        pygame.display.flip()       # lanza el dibujo a la pantalla
        pygame.time.wait(1000/30)   # ajusta para trabajar a 30 fps.

    # una vez fuera del loop, cierra la ventana
    pygame.quit()

#####################################################################
